/*
 * navigation.c
 *
 *  Created on: Apr 8, 2015
 *      Author: Jordan
 */

#include "sys_common.h"
#include "config.h"
#include "navigation.h"
#include "mpu6000.h"
#include "ms5611.h"
#include "hmc5883.h"
#include "GPS.h"
#include "math.h"
#include "EKF.h"
#include "board_io.h"

extern float32 AHRS_states[AHRS_N_STATES];
extern float32 INS_states[INS_N_STATES];

float32 wx, wy, wz;
float32 home_lat, home_lon, home_alt;

uint8 INS_enabled = 0;

uint8 get_INS_enabled(){
	return INS_enabled;
}

void AHRS_init(){
	float32 ax = mpu6000_get_acc('x') - ACC_X_OFFSET;
	float32 ay = mpu6000_get_acc('y') - ACC_Y_OFFSET;
	float32 az = mpu6000_get_acc('z') - ACC_Z_OFFSET;

	float32 acc_norm = sqrtf(powf(ax,2) + powf(ay,2) + powf(az,2));
	ax /= acc_norm;
	ay /= acc_norm;
	az /= acc_norm;

	hmc5883_update();
	float32 mag_x = hmc5883_get_mag('x');
	float32 mag_y = hmc5883_get_mag('y');
	float32 mag_z = hmc5883_get_mag('z');

	mag_x = (mag_x - MAG_X_OFFSET)*MAG_X_SCALE;
	mag_y = (mag_y - MAG_Y_OFFSET)*MAG_Y_SCALE;
	mag_z = (mag_z - MAG_Z_OFFSET)*MAG_Z_SCALE;

	float32 mag_norm = sqrtf(powf(mag_x,2) + powf(mag_y,2) + powf(mag_z,2));
	mag_x /= mag_norm;
	mag_y /= mag_norm;
	mag_z /= mag_norm;

	float32 phi = atan2f(ay, az);
	float32 theta = asinf(-ax);
	float32 psi = atan2f(-(cosf(phi)*mag_y-sinf(phi)*mag_z),cosf(theta)*mag_x+sinf(phi)*sinf(theta)*mag_y+cosf(phi)*sinf(theta)*mag_z);

	float32 qs = cosf(phi/2)*cosf(theta/2)*cosf(psi/2) + sinf(phi/2)*sinf(theta/2)*sinf(psi/2);
	float32 qx = sinf(phi/2)*cosf(theta/2)*cosf(psi/2) - cosf(phi/2)*sinf(theta/2)*sinf(psi/2);
	float32 qy = cosf(phi/2)*sinf(theta/2)*cosf(psi/2) + sinf(phi/2)*cosf(theta/2)*sinf(psi/2);
	float32 qz = cosf(phi/2)*cosf(theta/2)*sinf(psi/2) - sinf(phi/2)*sinf(theta/2)*cosf(psi/2);

	init_AHRS(qs, qx, qy, qz, 0, 0, 0);

	#if EKF_DEBUG == 1
		sci_printf("%0.5f,%0.5f,%0.5f,%0.5f\r\n",AHRS_states[0],AHRS_states[1],AHRS_states[2],AHRS_states[3]);
	#endif
}

void INS_init(){
	if(get_gps_fix() > 0){
		float32 alt_avg = 0;
		float32 speed_avg = 0;

		uint32 i;
		for(i=0;i<INIT_SAMPLE_LENGTH;i++){
			ms5611_update();
			alt_avg += ms5611_get_alt()/INIT_SAMPLE_LENGTH;
			speed_avg += get_gps_speed()/INIT_SAMPLE_LENGTH;
			delay_ms(20);
			toggle_LEDs(1,0,0,0);
		}

		home_lat = get_gps_lat()*PI/180;
		home_lon = get_gps_lon()*PI/180;
		home_alt = alt_avg*0.3048;

		float32 heading = get_gps_heading();

		float32 px = 0;
		float32 py = 0;
		float32 pz = 0;
		float32 vx = speed_avg*cosf(heading);
		float32 vy = speed_avg*sinf(heading);
		float32 vz = 0;

		init_INS(px, py, pz, vx, vy, vz, 0, 0, 0);

		INS_enabled = 1;
	} else {
		INS_enabled = 0;
	}
}

void navigation_step(){
	float32 ax = mpu6000_get_acc('x');
	float32 ay = mpu6000_get_acc('y');
	float32 az = mpu6000_get_acc('z');

	wx = mpu6000_get_gyro('x')*PI/180;
	wy = mpu6000_get_gyro('y')*PI/180;
	wz = mpu6000_get_gyro('z')*PI/180;

//	sci_printf("%7.4f,%7.4f,%7.4f,%7.4f,%7.4f,%7.4f\r\n", ax, ay, az, wx, wy, wz);

	ms5611_update();
	float32 bar_alt = ms5611_get_alt();

	hmc5883_update();
	float32 mag_x = hmc5883_get_mag('x');
	float32 mag_y = hmc5883_get_mag('y');
	float32 mag_z = hmc5883_get_mag('z');

	float32 lat = get_gps_lat();
	float32 lon = get_gps_lon();
	float32 speed = get_gps_speed();
	float32 heading = get_gps_heading();

	#if SENSOR_USE_RAW == 0
		wx -= GYRO_X_OFFSET;
		wy -= GYRO_Y_OFFSET;
		wz -= GYRO_Z_OFFSET;

		ax -= ACC_X_OFFSET;
		ay -= ACC_Y_OFFSET;
		az -= ACC_Z_OFFSET;

		mag_x = (mag_x - MAG_X_OFFSET)*MAG_X_SCALE;
		mag_y = (mag_y - MAG_Y_OFFSET)*MAG_Y_SCALE;
		mag_z = (mag_z - MAG_Z_OFFSET)*MAG_Z_SCALE;
	#endif

	update_AHRS_u(wx, wy, wz);
	update_AHRS_z(ax, ay, az, mag_x, mag_y, mag_z);
	AHRS_step();
	AHRS_normalize_q();

	float32 x_gps = 6371000*(lat*PI/180 - home_lat);
	float32 y_gps = 6371000*(lon*PI/180 - home_lon)*cosf((lat*PI/180 + home_lat)/2);
	float32 z_alt = -(bar_alt*0.3048 - home_alt);
	float32 x_dot_gps = speed*cosf(heading);
	float32 y_dot_gps = speed*sinf(heading);

	if(INS_enabled){
		update_INS_u(-ax*GRAVITY, -ay*GRAVITY, -az*GRAVITY);
		update_INS_z(x_gps, y_gps, z_alt, x_dot_gps, y_dot_gps);
		INS_step();
//		sci_printf("%7.4f,%7.4f,%7.4f,%7.4f,%7.4f,%7.4f,%7.4f,",AHRS_states[0], AHRS_states[1], AHRS_states[2], AHRS_states[3], AHRS_states[4], AHRS_states[5], AHRS_states[6]);
//		sci_printf("%7.4f,%7.4f,%7.4f,%7.4f,%7.4f,%7.4f,%7.4f,%7.4f,%7.4f\r\n",	INS_states[0], INS_states[1], INS_states[2], INS_states[3], INS_states[4], INS_states[5], INS_states[6], INS_states[7], INS_states[8]);
	}

	#if EKF_DEBUG == 1
		sci_printf("%0.2f,%0.2f,%0.2f,%0.2f,%0.2f,%0.2f,%0.5f,%0.5f,%0.5f,%0.5f,%0.5f,%0.5f,%0.5f\r\n", ax, ay, az, wx, wy, wz, AHRS_states[0],AHRS_states[1],AHRS_states[2],AHRS_states[3],AHRS_states[4],AHRS_states[5],AHRS_states[6]);
	#endif

	#if SENSOR_DEBUG == 1
		sci_printf("%7.4f,%7.4f,%7.4f,%7.4f,%7.4f,%7.4f,%7.4f,%7.4f,%7.4f,%11.6f,%11.6f,%6.2f,%5.2f,%5.2f\r\n", ax, ay, az, wx, wy, wz, mag_x, mag_y, mag_z, lat, lon, bar_alt, speed, heading);
	#endif

	#if EULER_DEBUG == 1
//		sci_printf("Roll: %5.2f, Pitch: %5.2f, P: %5.2f, Q: %5.2f, R: %5.2f\r\n", get_roll(), get_pitch(), get_roll_rate(), get_pitch_rate(), get_yaw_rate());
//		sci_printf("%5.2f,%5.2f,%5.2f,%5.2f,%5.2f\r\n", get_roll(), get_pitch(), get_roll_rate(), get_pitch_rate(), get_yaw_rate());
		sci_printf("+++RLL:%5.2f,PCH:%5.2f,COG:%5.2f***",get_roll()*180/PI, get_pitch()*180/PI, get_yaw()*180/PI);
	#endif
}

float32 get_roll(){
	float32 qs = AHRS_states[0];
	float32 qx = AHRS_states[1];
	float32 qy = AHRS_states[2];
	float32 qz = AHRS_states[3];
	return atan2f(2*(qy*qz+qx*qs), 1-2*(powf(qx,2)+powf(qy,2)));
}

float32 get_pitch(){
	float32 qs = AHRS_states[0];
	float32 qx = AHRS_states[1];
	float32 qy = AHRS_states[2];
	float32 qz = AHRS_states[3];
	return asinf(-2*(qx*qz-qy*qs));
}

float32 get_yaw(){
	float32 qs = AHRS_states[0];
	float32 qx = AHRS_states[1];
	float32 qy = AHRS_states[2];
	float32 qz = AHRS_states[3];
	return atan2f(2*(qx*qy+qz*qs), 1-2*(powf(qy,2)+powf(qz,2)));
}

float32 get_roll_rate(){
//	return wx - AHRS_states[4];
	return wx;
}

float32 get_pitch_rate(){
//	return wy - AHRS_states[5];
	return wy;
}

float32 get_yaw_rate(){
//	return wz - AHRS_states[6];
	return wz;
}

float32 get_NED_x(){
	return INS_states[0];
}

float32 get_NED_y(){
	return INS_states[1];
}

float32 get_NED_z(){
	return INS_states[2];
}

float32 get_NED_x_dot(){
	return INS_states[3];
}

float32 get_NED_y_dot(){
	return INS_states[4];
}

float32 get_NED_z_dot(){
	return INS_states[5];
}

float32 get_INS_lat(){
	return (get_NED_x()/6371000 + home_lat)*180/PI;
}

float32 get_INS_lon(){
	return (get_NED_y()/6371000/cosf((get_INS_lat()*PI/180 + home_lat)/2) + home_lon)*180/PI;
}

float32 get_nav_x_dot(){
	float32 psi = get_yaw();
	return cosf(psi)*get_NED_x_dot() + sinf(psi)*get_NED_y_dot();
}

float32 get_nav_y_dot(){
	float32 psi = get_yaw();
	return -sinf(psi)*get_NED_x_dot() + cosf(psi)*get_NED_y_dot();
}

float32 get_wpt_x(float32 lat){
	return 6371000*(lat*PI/180 - home_lat);
}

float32 get_wpt_y(float32 lat, float32 lon){
	return 6371000*(lon*PI/180 - home_lon)*cosf((lat*PI/180 + home_lat)/2);
}
